;I2Cmastd.asm john waller 2002oct16 to experiment with master end of I2C MSSP.
;Transfer of two bytes slave to master with handshake, interrupt, and error checking.

#DEFINE BANK0   BCF $03,5   ;STATUS bit 5
#DEFINE BANK1   BSF $03,5   ;STATUS bit 5

INDF:    .EQU $00	;Indirect address, bank 0, 1, 2, 3.
TMRO:	 .EQU $01	;Timer 0 value, bank 0, 3.
OPTION:  .EQU $01	;Option register, bank 1, 3.
PCL:     .EQU $02	;Program counter low byte, bank 0, 1, 2, 3.
STATUS:  .EQU $03	;Status flags, bank 0, 1, 2, 3.
FSR:     .EQU $04	;Indirect address pointer, bank 0, 1, 2, 3.
PORTA:   .EQU $05	;Port A, bank 0.
TRISA:   .EQU $05	;Port A direction register, bank 1.
PORTB:   .EQU $06	;Port B, bank 0, 2.
TRISB:   .EQU $06	;Port B direction register, bank 1, 3.
PORTC:   .EQU $07	;Port C, bank 0.
TRISC:   .EQU $07	;Port C direction register, bank 1.
PORTD:   .EQU $08	;Port D, bank 0.
TRISD:   .EQU $08	;Port D direction register, bank 1.
PORTE:   .EQU $09	;Port E, bank 0.
TRISE:   .EQU $09	;Port E direction register, bank 1.
PCLATH:	 .EQU $0A	;Program counter high byte setting register, bank 0, 1, 2, 3.
INTCON:  .EQU $0B	;Interrupt control, bank 0, 1, 2, 3.
PIR1:	 .EQU $0C	;Peripheral interrupt flags, bank 0.
PIE1:    .EQU $0C	;Peripheral interrupt enable, bank 1.
PIR2:    .EQU $0D	;Peripheral interrupt flags, bank 0.
PIE2:    .EQU $0D	;Peripheral interrupt enable, bank 1.
SSPCON2: .EQU $11	;SSP configuration, bank 1.
SSPBUF:	 .EQU $13	;SSP address and data buffer, bank 0.
SSPADD:  .EQU $13	;SSP master baud rate/slave address, bank 1.
SSPCON:  .EQU $14	;SSP configuration, bank 0.
SSPSTAT: .EQU $14	;SSP status, bank 1.
ADCON1:	 .EQU $1F	;Port A A/D control register, bank 1.

;........GENERAL PURPOSE REGISTERS

STBTCT:  .EQU $20	;SSP transmit byte count, bank 0.

			;Registers to save working, FSR, STATUS, and PCLATH at interrupt.
			;Must be visible across all banks, 70H..7FH.
FSRINT:	.EQU $7C	;For FSR register, alias, FCH, 17CH, & 1FCH.
PCLINT:	.EQU $7D	;For PCLATH register, alias, FDH, 17DH, & 1FDH.
STAINT:	.EQU $7E	;For STATUS register, alias, FEH, 17EH, & 1FEH.
WRKINT:	.EQU $7F	;For working register, alias FFH, 17FH, & 1FFH.

;........CONSTANT VALUES

ERRINT:	.EQU $20	;Tag for an interrupt error.
ERRSAD:	.EQU $40	;Tag for a master address error.
ERRSAK:	.EQU $60	;Tag for a failure of slave to acknowledge.
ERRSDT:	.EQU $80	;Tag for a master data load error.
SLVADD: .EQU $10	;Slave address; upper 7 bits; equate must be even;
			;do not use F8h, FAh, FCh, FEh, or zero.
			;16F877 manual says only 112 valid addresses available;
			;don't know what the other invalid ones are.

;........BIT VALUES

ACKDT:  .EQU 5		;SSP master acknowledge data.
ACKEN:  .EQU 4		;SSP master 
ACKSTAT:.EQU 6		;SSP master acknowledge status.
BCLIE   .EQU 3		;Bus collision interrupt enable.
BCLIF:  .EQU 3		;Bus collision interrupt flag.
BF:     .EQU 0		;SSP buffer full.
C:      .EQU 0          ;Carry status.
CKE:    .EQU 6		;Clock edge select.
CKP:	.EQU 4		;Hold/enable SSP clock pulse by slave.
DNA     .EQU 5		;Last byte received or transmitted was data or address.
F:      .EQU 1          ;Select file register.
GCEN:   .EQU 7		;SSP general call enable.
GIE:    .EQU 7		;General interrupt enable.
P:      .EQU 4		;Stop bit.
PEIE:   .EQU 6		;Peripheral interrupt enable.
PEN:    .EQU 2		;SSP master stop enable.
RCEN:   .EQU 3		;SSP receive enable.
RNW:    .EQU 2		;Read/write information following the last address match.
RP0:    .EQU 5          ;STATUS bank control bit.
RP1:    .EQU 6          ;STATUS bank control bit.
RSEN:	.EQU 1		;SSP master repeat start enable.
SEN:    .EQU 0		;SSP master start enable.
SMP:    .EQU 7		;Slew rate control.
SSPIE:  .EQU 3		;SSP interrupt enable.
SSPIF:  .EQU 3		;SSP interrupt flag.
SSPEN:  .EQU 5		;Configure SDA and SCL pins for SSP.
SSPOV:  .EQU 6		;SSP receive overflow detect.
UA:     .EQU 1		;Update address; 10-bit address mode only.
W:      .EQU 0          ;Select working register.
WCOL:   .EQU 7		;SSP write collision detect.
Z:      .EQU 2          ;Zero status.

;..........

        .ORG $0004
	goto INTERRUPT
        .ORG $0005

	goto INITIALISE	;jump over sub-page 0 tables to INITIALISE.

		;page 0, sub-page 0 tables: make sure they don't overrun.
		
;..........

BACKGROUND:	;Main part of program in background; entered from INITIALISE.
		;Called from none; calls none.
BACK1:	btfsc PORTA,4		;Loop until
	goto BACK1		;switch closed.
	clrf STBTCT		;Clear transmit
	BANK1			;byte count &
	bsf SSPCON2,SEN		;start
	BANK0			;transmission.
BACK2:	goto BACK2		;Pause.
	goto BACKGROUND		;Loop back.

;..........

CHECKANDSTOP:	;Writes selected registers to ports and pauses.
		;Called from ERRORINT, ERRORSACKT, ERRORSADDT; calls, none.
	BANK0			;
	andlw %11100000		;Write high bits in W to port C
	movwf PORTC		;to indicate where in program stop occurred.
	bsf PORTE,2		;Signal program has reached here.
	btfss PIR1,SSPIF	;Show SSP
	goto CHAST1		;interrupt
	bsf PORTC,2		;bit
	goto CHAST2		;on
CHAST1:	bcf PORTC,2		;port C,2.
CHAST2:	BANK1			;Write
	movf SSPSTAT,W		;register
	BANK0			;contents
	movwf PORTB		;to port.
	movf SSPCON,W		;Write register
	movwf PORTD		;contents to port.
CHAST3:	goto CHAST3		;Loop indefinitely.

;..........

ERRORINT:	;Sets tag for an interrupt error and halts execution.
		;Called from INTERRUPT; calls CHECKANDSTOP.
	movlw ERRINT		;Set interrupt error tag, write SSPSTAT and SSPCON2 to
	call CHECKANDSTOP	;ports B and D, respectively, and halt.

;..........

ERRORSACKT:	;Sets tag for a failure of slave to acknowledge with ACK.
		;Called from HANDSSPACKT; calls CHECKANDSTOP.
	movlw ERRSAK		;Set slave acknowledge failure tag, write SSPSTAT
				;and SSPCON to ports B and D,
	call CHECKANDSTOP	;respectively, and halt.

;..........

ERRORSADDT:	;Sets tag for a master transmit address error and halts execution.
		;Called from HANDSSPADDT; calls CHECKANDSTOP.
	movlw ERRSAD		;Set master transmit address error tag, write SSPSTAT
				;and SSPCON to ports B and D,
	call CHECKANDSTOP	;respectively, and halt.

;..........

HANDSSPACKT:	;Handles an acknowledgment from slave and halts execution if not ACK.
		;Called from INTERRUPT; calls ERRORSACKT.
	BANK1			;
	btfsc SSPCON2,ACKSTAT	;Skip if ACK was received,
	call ERRORSACKT		;otherwise report error and halt.
	BANK0			;
	return			;
	
;..........

HANDSSPADDT:	;Handles a start complete interrupt by sending a slave address and
		;checking for a write collision error.
		;Called from INTERRUPT; calls ERROSADDT.
	movlw SLVADD		;Write slave address to SSP buffer
	iorlw %00000001		;with read from slave bit set
	movwf SSPBUF		;to start transmission.
	btfsc SSPCON,WCOL	;Skip if no write collision,
	call ERRORSADDT		;otherwise report error and halt.
	incf STBTCT,F		;Increment byte count.
	return			;
	
;..........

HANDSSPRBFS:	;Handles a read byte from slave interrupt.
		;Called from INTERRUPT; calls none.
	movf STBTCT,W		;Skip if
	sublw 2			;first byte
	btfss STATUS,Z		;read.
	goto HRBFS1		;
	movf SSPBUF,W		;Put buffer
	movwf PORTB		;contents in
	incf STBTCT,F		;port and increment
	return			;byte count.
HRBFS1: movf SSPBUF,W		;Put buffer
	movwf PORTD		;contents in
	incf STBTCT,F		;port and increment
	return			;byte count.
	
;..........

HANDSSPSATS:	;Handles interrupt to send ACK to slave.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bcf SSPCON2,ACKDT	;Send ACK
	bsf SSPCON2,ACKEN	;to slave.
	BANK0			;
	return			;
	
;..........

HANDSSPSNTS:	;Handles interrupt to send NACK to slave.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bsf SSPCON2,ACKDT	;Send NACK
	bsf SSPCON2,ACKEN	;to slave.
	BANK0			;
	return			;
	
;..........

HANDSSPSRBF:	;Handles interrupt where read byte flag is to be set.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bsf SSPCON2,RCEN	;Set flag.
	BANK0			;
	incf STBTCT,F		;Increment byte count.
	return			;
	
;..........

HANDSSPSTOP:	;Handles an interrupt where 'stop' is to be initiated.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bsf SSPCON2,PEN		;Initiate 'stop'.
	BANK0			;
	incf STBTCT,F		;Increment byte count. 
	return			;
	
;..........

INITI2CMSTR:	;Initialises the I2C master device.
		;Called from INITIALISE; calls none.
	movlw %00101000		;Bits 7, 6, flags; bit 5 enable SDA and SCL port pins;
	movwf SSPCON		;bit 4, unused; bits 3..0, I2C master mode.
	BANK1			;
	movlw %10000000		;Bit 7, slew rate control disabled; bit 6, comply with
	movwf SSPSTAT		;I2C protocol; bits 5..0, flags.
	clrf SSPCON2		;All bits flags.
	movlw 24		;Set baud rate to 100 kHz 
	movwf SSPADD		;for 10 MHz oscillator.
	bsf PIE1,SSPIE		;Enable SSP interrupt.
	BANK0			;
	bsf INTCON,PEIE		;Enable peripheral interrupts.
	bsf INTCON,GIE		;Enable general interrupts.
	return			;
		
;..........

INITIALISE:	;Initialising ports and others functions. Entered at startup.
		;Called from INITI2CMSTR; calls none.
	clrf PCLATH		;Page and sub-page 0.
	clrf STATUS		;Bank 0 and clear all flags.
				;Clear all gpr locations in banks 0 and 1.
	movlw $20		;First gpr address in bank 0
	movwf FSR		;to indirect address register.
INITB0:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss FSR,7		;Skip if end of bank 0.
	goto INITB0		;
	movlw $A0		;First gpr address in bank 1
	movwf FSR		;to indirect address register.
INITB1:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss STATUS,Z		;Skip if end of bank 1.
	goto INITB1		;
				;Set up ports and timers.
	clrf PORTA		;Clear port.
        clrf PORTB		;Clear port.
	clrf PORTC		;Clear port.
        clrf PORTD		;Clear port.
	clrf PORTE		;Clear port.
        BANK1			;
	movlw %00000110		;Port A as digital
	movwf ADCON1		;input/output.
	movlw %00111111		;Port A0-A5
        movwf TRISA     	;as input.
        clrf TRISB      	;Port B0-B7 as output.
	movlw %00011000		;Port C all bits output, except
	movwf TRISC		;3 and 4 must be input for I2C.
	movlw %00000000		;Port D bits all
	movwf TRISD		;outputs.
	movlw %00000000		;Port E all bits as output,
	movwf TRISE		;disable slave port.
        BANK0			;
        clrf INTCON		;No interrupts or timers for the first try.
        call INITI2CMSTR	;Initialise naster SSP.
        goto BACKGROUND		;
        
	;...........

INTERRUPT:	;Interrupt routine goes here.
		;Called from INITIALISE; calls ERRORINT, HANDSSPADDT, HANDSSPACKT,
		;HANDSSPSRBF, HANDSSPRBFS, HANDSSPSATS, HANDSSPSNTS, HANDSSPSTOP.
				;Save registers and set up to execute interrupt.
	movwf WRKINT		;Save working register.
	swapf STATUS,W		;Save status
	movwf STAINT		;register, without affecting flags.
	movf PCLATH,W		;Save page
	movwf PCLINT		;register.
	movf FSR,W		;Save indirect
	movwf FSRINT		;address register.
	clrf STATUS		;Set bank, page,
	clrf PCLATH		;and sub-page 0.
	
				;Execute interrupt.
	btfss PIR1,SSPIF	;If not SSP interrupt, call error
	call ERRORINT		;routine and halt execution, otherwise service
	bcf PIR1,SSPIF		;interrupt, first clearing interrupt flag.
				;
	movf STBTCT,W		;If byte count zero
	btfss STATUS,Z		;'start' is complete;
	goto INTER1		;send address
	call HANDSSPADDT	;to slave; increment byte count.
	goto INTEND		;
INTER1:	movf STBTCT,W		;If byte
	sublw 1			;count one,
	btfss STATUS,Z		;check for
	goto INTER2		;acknowledgment
	call HANDSSPACKT	;and set read
	call HANDSSPSRBF	;byte flag.
	goto INTEND		;	
INTER2:	movf STBTCT,W		;If byte
	sublw 2			;count two,
	btfss STATUS,Z		;read buffer
	goto INTER3		;contents
	call HANDSSPRBFS	;and send
	call HANDSSPSATS	;ACK to slave.
	goto INTEND		;	
INTER3:	movf STBTCT,W		;If byte
	sublw 3			;count three,
	btfss STATUS,Z		;set read
	goto INTER4		;byte
	call HANDSSPSRBF	;flag.
	goto INTEND		;
INTER4:	movf STBTCT,W		;If byte
	sublw 4			;count four,
	btfss STATUS,Z		;read buffer
	goto INTER5		;contents
	call HANDSSPRBFS	;and send
	call HANDSSPSNTS	;NACK to slave.
	goto INTEND		;
INTER5:	movf STBTCT,W		;If byte
	sublw 5			;count five,
	btfss STATUS,Z		;initiate
	goto INTER6		;a
	call HANDSSPSTOP	;'stop'.
	goto INTEND		;	
INTER6:	movf STBTCT,W		;If byte
	sublw 6			;count six,
	btfss STATUS,Z		;'stop' is complete;
	clrf STBTCT		;clear byte count.

INTEND:				;Restore registers and return.
	movf FSRINT,W		;Restore indirect
	movwf FSR		;address register.
	movf PCLINT,W		;Restore page
	movwf PCLATH		;register.
	swapf STAINT,W		;Restore status
	movwf STATUS		;register.
	swapf WRKINT,F		;Restore working
	swapf WRKINT,W		;register.
	retfie			;

;..........

        .END
